<?php

namespace App\Console\Commands;

use App\Models\CoachPlanPurchase;
use App\Models\User;
use App\Notifications\CoachPlanExpiringSoon;
use App\Notifications\CoachPlanExpired;
use Illuminate\Console\Command;

class ProcessCoachPlanExpirations extends Command
{
    protected $signature = 'coach-plans:process-expirations';
    protected $description = 'Send notifications for expiring and expired coach plan purchases';

    public function handle(): int
    {
        $this->info('Processing coach plan expirations...');

        // Find plans expiring in 7 days
        $this->notifyExpiringSoon(7);

        // Find plans expiring in 3 days
        $this->notifyExpiringSoon(3);

        // Find plans expiring in 1 day
        $this->notifyExpiringSoon(1);

        // Find plans that just expired (today)
        $this->notifyExpired();

        $this->info('Coach plan expirations processed successfully.');

        return Command::SUCCESS;
    }

    private function notifyExpiringSoon(int $days): void
    {
        $purchases = CoachPlanPurchase::where('status', 'paid')
            ->whereDate('ends_at', now()->addDays($days)->toDateString())
            ->whereNull('expiring_notified_at')
            ->with(['coachPlan', 'user'])
            ->get();

        foreach ($purchases as $purchase) {
            if ($purchase->user) {
                $purchase->user->notify(new CoachPlanExpiringSoon($purchase, $days));
            } elseif ($purchase->buyer_email) {
                // Create a temporary user for notification
                $tempUser = new User(['email' => $purchase->buyer_email, 'name' => $purchase->buyer_name]);
                $tempUser->notify(new CoachPlanExpiringSoon($purchase, $days));
            }

            // Mark as notified only for 7-day reminder to avoid duplicate notifications
            if ($days === 7) {
                $purchase->update(['expiring_notified_at' => now()]);
            }

            $this->line("  - Notified {$purchase->buyer_name} about plan expiring in {$days} days");
        }

        $this->info("Sent {$purchases->count()} expiring soon notifications ({$days} days)");
    }

    private function notifyExpired(): void
    {
        $purchases = CoachPlanPurchase::where('status', 'paid')
            ->whereDate('ends_at', now()->subDay()->toDateString())
            ->whereNull('expired_notified_at')
            ->with(['coachPlan', 'user'])
            ->get();

        foreach ($purchases as $purchase) {
            if ($purchase->user) {
                $purchase->user->notify(new CoachPlanExpired($purchase));
            } elseif ($purchase->buyer_email) {
                $tempUser = new User(['email' => $purchase->buyer_email, 'name' => $purchase->buyer_name]);
                $tempUser->notify(new CoachPlanExpired($purchase));
            }

            $purchase->update([
                'expired_notified_at' => now(),
                'status' => 'expired',
            ]);

            $this->line("  - Notified {$purchase->buyer_name} about expired plan");
        }

        $this->info("Sent {$purchases->count()} expired notifications");
    }
}
