<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use App\Notifications\OrderStatusChanged;
use App\Notifications\OrderShipped;
use App\Notifications\OrderDelivered;
use App\Notifications\OrderCancelled;
use App\Notifications\OrderRefunded;
use App\Notifications\OrderPaid;
use App\Notifications\AdminOrderShipped;
use App\Notifications\AdminOrderDelivered;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Inertia\Inertia;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $query = Order::with(['user', 'items'])
            ->when($request->search, function ($q, $search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            })
            ->when($request->status, function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->from, function ($q, $from) {
                $q->whereDate('created_at', '>=', $from);
            })
            ->when($request->to, function ($q, $to) {
                $q->whereDate('created_at', '<=', $to);
            })
            ->latest();

        $stats = [
            'total' => Order::count(),
            'pending' => Order::pending()->count(),
            'processing' => Order::processing()->count(),
            'shipped' => Order::shipped()->count(),
            'delivered' => Order::delivered()->count(),
            'revenue' => Order::where('payment_status', 'paid')->sum('total'),
        ];

        return Inertia::render('Admin/Store/Orders/Index', [
            'orders' => $query->paginate(15)->withQueryString(),
            'stats' => $stats,
            'filters' => $request->only(['search', 'status', 'from', 'to']),
        ]);
    }

    public function show(Order $order)
    {
        $order->load(['user', 'items.product']);

        return Inertia::render('Admin/Store/Orders/Show', [
            'order' => $order,
        ]);
    }

    public function updateStatus(Request $request, Order $order)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,processing,shipped,delivered,cancelled,refunded',
            'tracking_number' => 'nullable|string|max:255',
            'tracking_url' => 'nullable|url|max:500',
            'admin_notes' => 'nullable|string',
            'cancellation_reason' => 'nullable|string|max:500',
            'refund_amount' => 'nullable|numeric|min:0',
        ]);

        $oldStatus = $order->status;
        $newStatus = $validated['status'];

        // Update order status
        $order->updateStatus($newStatus);

        if (isset($validated['tracking_number'])) {
            $order->tracking_number = $validated['tracking_number'];
        }
        if (isset($validated['tracking_url'])) {
            $order->tracking_url = $validated['tracking_url'];
        }
        if (isset($validated['admin_notes'])) {
            $order->admin_notes = $validated['admin_notes'];
        }
        $order->save();

        // Send appropriate notification based on status change
        $this->sendStatusNotification($order, $oldStatus, $newStatus, $validated);

        return back()->with('success', 'Estado del pedido actualizado.');
    }

    /**
     * Send the appropriate notification based on status change.
     */
    protected function sendStatusNotification(Order $order, string $oldStatus, string $newStatus, array $data): void
    {
        // Don't notify if status didn't change
        if ($oldStatus === $newStatus) {
            return;
        }

        $user = $order->user;
        
        // Get all admins for admin notifications
        $admins = User::where('role', 'admin')->get();

        switch ($newStatus) {
            case 'shipped':
                // Notify customer
                $user->notify(new OrderShipped($order));
                // Notify admins (except the one who made the change)
                $otherAdmins = $admins->filter(fn($admin) => $admin->id !== auth()->id());
                if ($otherAdmins->isNotEmpty()) {
                    Notification::send($otherAdmins, new AdminOrderShipped($order));
                }
                break;

            case 'delivered':
                // Notify customer
                $user->notify(new OrderDelivered($order));
                // Notify admins (except the one who made the change)
                $otherAdmins = $admins->filter(fn($admin) => $admin->id !== auth()->id());
                if ($otherAdmins->isNotEmpty()) {
                    Notification::send($otherAdmins, new AdminOrderDelivered($order));
                }
                break;

            case 'cancelled':
                $user->notify(new OrderCancelled(
                    $order,
                    $data['cancellation_reason'] ?? null
                ));
                break;

            case 'refunded':
                $refundAmount = $data['refund_amount'] ?? $order->total;
                $user->notify(new OrderRefunded(
                    $order,
                    $refundAmount,
                    $data['cancellation_reason'] ?? null
                ));
                break;

            default:
                // For other status changes, send generic notification
                $user->notify(new OrderStatusChanged($order, $oldStatus, $newStatus));
                break;
        }
    }

    public function markAsPaid(Order $order)
    {
        $oldStatus = $order->status;
        $order->markAsPaid();

        // Notify customer that payment was marked as received
        $order->user->notify(new OrderPaid($order));

        return back()->with('success', 'Pedido marcado como pagado.');
    }

    public function destroy(Order $order)
    {
        $order->delete();

        return redirect()->route('admin.store.orders.index')
            ->with('success', 'Pedido eliminado.');
    }

    public function export(Request $request)
    {
        // TODO: Implement CSV/Excel export
        return back()->with('success', 'Exportación iniciada.');
    }

    public function bulkCancel(Request $request)
    {
        $validated = $request->validate([
            'order_ids' => 'required|array',
            'order_ids.*' => 'exists:orders,id',
        ]);

        $orders = Order::whereIn('id', $validated['order_ids'])->get();
        $count = 0;

        foreach ($orders as $order) {
            if (!in_array($order->status, ['delivered', 'cancelled', 'refunded'])) {
                $order->updateStatus('cancelled');
                $count++;
            }
        }

        return back()->with('success', "{$count} pedido(s) cancelado(s) exitosamente.");
    }

    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'order_ids' => 'required|array',
            'order_ids.*' => 'exists:orders,id',
        ]);

        $count = Order::whereIn('id', $validated['order_ids'])->delete();

        return back()->with('success', "{$count} pedido(s) eliminado(s) exitosamente.");
    }
}
