<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Faq;
use App\Models\Setting;
use Illuminate\Http\Request;
use Inertia\Inertia;

class SupportSettingsController extends Controller
{
    /**
     * Display support settings page.
     */
    public function index()
    {
        return Inertia::render('Admin/Support/Index', [
            'faqs' => Faq::ordered()->get(),
            'supportEmail' => Setting::get('support_email', 'soporte@coachpro.com'),
        ]);
    }

    /**
     * Update support email.
     */
    public function updateEmail(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|email|max:255',
        ]);

        Setting::set('support_email', $validated['email'], 'support');

        return back()->with('success', 'Email de soporte actualizado');
    }

    /**
     * Store a new FAQ.
     */
    public function storeFaq(Request $request)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:500',
            'answer' => 'required|string|max:5000',
        ]);

        $maxOrder = Faq::max('order') ?? 0;
        
        Faq::create([
            'question' => $validated['question'],
            'answer' => $validated['answer'],
            'order' => $maxOrder + 1,
        ]);

        return back()->with('success', 'Pregunta frecuente creada');
    }

    /**
     * Update an existing FAQ.
     */
    public function updateFaq(Request $request, Faq $faq)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:500',
            'answer' => 'required|string|max:5000',
            'is_active' => 'boolean',
        ]);

        $faq->update($validated);

        return back()->with('success', 'Pregunta frecuente actualizada');
    }

    /**
     * Delete a FAQ.
     */
    public function destroyFaq(Faq $faq)
    {
        $faq->delete();

        return back()->with('success', 'Pregunta frecuente eliminada');
    }

    /**
     * Reorder FAQs.
     */
    public function reorderFaqs(Request $request)
    {
        $validated = $request->validate([
            'order' => 'required|array',
            'order.*' => 'integer|exists:faqs,id',
        ]);

        foreach ($validated['order'] as $index => $id) {
            Faq::where('id', $id)->update(['order' => $index]);
        }

        return back()->with('success', 'Orden actualizado');
    }
}
