<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\LandingTestimonial;
use App\Models\MealPlan;
use App\Models\Message;
use App\Models\Workout;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Inertia\Inertia;
use Inertia\Response;

/**
 * Controller for the Coach Dashboard
 */
class DashboardController extends Controller
{
    /**
     * Display the coach dashboard with stats and recent activity.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();

        // Get recent clients with their portal user avatar
        $recentClients = $user->clients()
            ->with('portalUser:id,avatar')
            ->latest()
            ->take(5)
            ->get(['id', 'name', 'email', 'is_active', 'client_user_id']);

        // Add avatar_url from portal user
        $recentClients->transform(function ($client) {
            $avatar = $client->portalUser?->avatar;
            if ($avatar) {
                $client->avatar_url = str_starts_with($avatar, 'http') || str_starts_with($avatar, '/storage/')
                    ? $avatar
                    : '/storage/' . $avatar;
            } else {
                $client->avatar_url = null;
            }
            return $client;
        });

        // Calculate stats - workouts and meal plans belong to clients, not directly to users
        $clientsCount = $user->clients()->count();
        $clientIds = $user->clients()->pluck('id');

        $workoutsCount = Workout::whereIn('client_id', $clientIds)->count();
        $mealPlansCount = MealPlan::whereIn('client_id', $clientIds)->count();
        $messagesCount = $user->sentMessages()->count() + $user->receivedMessages()->count();
        $unreadMessages = $user->receivedMessages()->whereNull('read_at')->count();

        // Get recent activity
        $recentActivity = $this->getRecentActivity($user, $clientIds);

        // Check if coach can submit a testimonial
        $canSubmitTestimonial = $this->canSubmitTestimonial($user);
        $testimonialStatus = $this->getTestimonialStatus($user);

        return Inertia::render('Coach/Dashboard', [
            'subscription' => $user->activeSubscription?->load('plan'),
            'stats' => [
                'clients' => $clientsCount,
                'remaining_clients' => $user->remaining('clients'),
                'workouts' => $workoutsCount,
                'mealPlans' => $mealPlansCount,
                'messages' => $messagesCount,
                'unreadMessages' => $unreadMessages,
            ],
            'recentClients' => $recentClients,
            'pendingTasks' => $recentActivity,
            'canSubmitTestimonial' => $canSubmitTestimonial,
            'testimonialStatus' => $testimonialStatus,
        ]);
    }

    /**
     * Get recent activity for the coach (last 7 days).
     */
    protected function getRecentActivity($user, Collection $clientIds): array
    {
        $activities = collect();
        $since = Carbon::now()->subDays(7);

        // Recent clients added
        $recentClientsAdded = Client::where('user_id', $user->id)
            ->where('created_at', '>=', $since)
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get(['id', 'name', 'created_at']);

        foreach ($recentClientsAdded as $client) {
            $activities->push([
                'id' => 'client_' . $client->id,
                'type' => 'client',
                'description' => "Nuevo cliente: {$client->name}",
                'date' => $client->created_at->diffForHumans(),
                'timestamp' => $client->created_at,
            ]);
        }

        // Recent workouts created
        if ($clientIds->isNotEmpty()) {
            $recentWorkouts = Workout::whereIn('client_id', $clientIds)
                ->where('created_at', '>=', $since)
                ->with('client:id,name')
                ->orderBy('created_at', 'desc')
                ->take(10)
                ->get(['id', 'name', 'client_id', 'created_at']);

            foreach ($recentWorkouts as $workout) {
                $activities->push([
                    'id' => 'workout_' . $workout->id,
                    'type' => 'workout',
                    'description' => "Rutina creada: {$workout->name}" . ($workout->client ? " para {$workout->client->name}" : ''),
                    'date' => $workout->created_at->diffForHumans(),
                    'timestamp' => $workout->created_at,
                ]);
            }

            // Recent meal plans created
            $recentMealPlans = MealPlan::whereIn('client_id', $clientIds)
                ->where('created_at', '>=', $since)
                ->with('client:id,name')
                ->orderBy('created_at', 'desc')
                ->take(10)
                ->get(['id', 'name', 'client_id', 'created_at']);

            foreach ($recentMealPlans as $mealPlan) {
                $activities->push([
                    'id' => 'meal_' . $mealPlan->id,
                    'type' => 'meal',
                    'description' => "Plan de comida: {$mealPlan->name}" . ($mealPlan->client ? " para {$mealPlan->client->name}" : ''),
                    'date' => $mealPlan->created_at->diffForHumans(),
                    'timestamp' => $mealPlan->created_at,
                ]);
            }
        }

        // Recent messages sent
        $recentMessages = Message::where('sender_id', $user->id)
            ->where('created_at', '>=', $since)
            ->with('receiver:id,name')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get(['id', 'receiver_id', 'created_at']);

        foreach ($recentMessages as $message) {
            $activities->push([
                'id' => 'message_' . $message->id,
                'type' => 'message',
                'description' => 'Mensaje enviado' . ($message->receiver ? " a {$message->receiver->name}" : ''),
                'date' => $message->created_at->diffForHumans(),
                'timestamp' => $message->created_at,
            ]);
        }

        // Sort by timestamp and take top 10
        return $activities
            ->sortByDesc('timestamp')
            ->take(10)
            ->map(function ($activity) {
                unset($activity['timestamp']);
                return $activity;
            })
            ->values()
            ->toArray();
    }

    /**
     * Check if coach can submit a new testimonial.
     */
    protected function canSubmitTestimonial($user): bool
    {
        // Check if user has a pending or approved testimonial
        return !LandingTestimonial::where('coach_id', $user->id)
            ->whereIn('status', ['pending', 'approved'])
            ->exists();
    }

    /**
     * Get the current testimonial status for the coach.
     */
    protected function getTestimonialStatus($user): ?string
    {
        $testimonial = LandingTestimonial::where('coach_id', $user->id)
            ->latest()
            ->first();

        return $testimonial?->status;
    }
}
