<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Models\Food;
use Illuminate\Http\Request;
use Inertia\Inertia;

/**
 * FoodController
 * Manages coach's food library
 */
class FoodController extends Controller
{
    /**
     * Display a listing of foods
     */
    public function index(Request $request)
    {
        $coach = auth()->user();
        
        $query = Food::query()
            ->where('is_active', true) // Only show active foods
            ->where(function ($q) use ($coach) {
                $q->whereNull('user_id') // System foods (global)
                  ->orWhere('user_id', $coach->id); // Coach's own foods
            });
        
        // Search filter
        if ($request->search) {
            $query->where('name', 'like', "%{$request->search}%");
        }
        
        // Category filter
        if ($request->category) {
            $query->where('category', $request->category);
        }

        // Meal type filter
        if ($request->meal_type) {
            $query->where('meal_type', $request->meal_type);
        }
        
        $foods = $query->orderBy('name')->paginate(20)->withQueryString();
        
        $categories = Food::distinct()->pluck('category')->filter();
        
        return Inertia::render('Coach/Foods/Index', [
            'foods' => $foods,
            'filters' => $request->only(['search', 'category', 'meal_type']),
            'categories' => $categories,
            'mealTypes' => Food::MEAL_TYPES,
        ]);
    }
    
    /**
     * Show form for creating a new food
     */
    public function create(Request $request)
    {
        $copyFood = null;
        if ($request->has('copy')) {
            $copyFood = Food::find($request->copy);
            if ($copyFood) {
                $copyFood = $copyFood->only([
                    'name', 'category', 'meal_type', 'serving_size', 
                    'serving_weight', 'calories', 'proteins', 'carbs', 
                    'fats', 'fiber'
                ]);
                $copyFood['name'] = $copyFood['name'] . ' (Copia)';
            }
        }
        
        return Inertia::render('Coach/Foods/Create', [
            'categories' => Food::distinct()->pluck('category')->filter(),
            'mealTypes' => Food::MEAL_TYPES,
            'copyFood' => $copyFood,
        ]);
    }
    
    /**
     * Store a newly created food
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'category' => 'nullable|string|max:100',
            'meal_type' => 'nullable|string|in:' . implode(',', array_keys(Food::MEAL_TYPES)),
            'serving_size' => 'nullable|string|max:100',
            'serving_weight' => 'nullable|numeric|min:0',
            'calories' => 'required|numeric|min:0',
            'proteins' => 'required|numeric|min:0',
            'carbs' => 'required|numeric|min:0',
            'fats' => 'required|numeric|min:0',
            'fiber' => 'nullable|numeric|min:0',
        ]);
        
        $validated['user_id'] = auth()->id();
        $validated['is_custom'] = true;
        $validated['is_active'] = true;
        
        Food::create($validated);
        
        return redirect()->route('coach.foods.index')
            ->with('success', 'Alimento creado correctamente');
    }
    
    /**
     * Show form for editing a food
     */
    public function edit(Food $food)
    {
        // Only allow editing own foods
        if ($food->user_id && $food->user_id !== auth()->id()) {
            abort(403);
        }
        
        return Inertia::render('Coach/Foods/Edit', [
            'food' => $food,
            'categories' => Food::distinct()->pluck('category')->filter(),
            'mealTypes' => Food::MEAL_TYPES,
        ]);
    }
    
    /**
     * Update the specified food
     */
    public function update(Request $request, Food $food)
    {
        // Only allow updating own foods
        if ($food->user_id && $food->user_id !== auth()->id()) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'category' => 'nullable|string|max:100',
            'meal_type' => 'nullable|string|in:' . implode(',', array_keys(Food::MEAL_TYPES)),
            'serving_size' => 'nullable|string|max:100',
            'serving_weight' => 'nullable|numeric|min:0',
            'calories' => 'required|numeric|min:0',
            'proteins' => 'required|numeric|min:0',
            'carbs' => 'required|numeric|min:0',
            'fats' => 'required|numeric|min:0',
            'fiber' => 'nullable|numeric|min:0',
        ]);
        
        $food->update($validated);
        
        return redirect()->route('coach.foods.index')
            ->with('success', 'Alimento actualizado correctamente');
    }
    
    /**
     * Remove the specified food
     */
    public function destroy(Food $food)
    {
        // Only allow deleting own foods
        if (!$food->user_id || $food->user_id !== auth()->id()) {
            abort(403, 'No puedes eliminar alimentos del sistema');
        }
        
        $food->delete();
        
        return back()->with('success', 'Alimento eliminado');
    }
}
