<?php

namespace App\Http\Controllers;

use App\Models\CoachPlan;
use App\Models\CoachPlanPurchase;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Inertia\Inertia;
use Inertia\Response;

class PublicCoachPlanController extends Controller
{
    /**
     * Display a public coach plan page.
     */
    public function show(string $slug): Response
    {
        $plan = CoachPlan::where('slug', $slug)
            ->where('is_active', true)
            ->where('is_public', true)
            ->with(['coach:id,name,email,avatar,bio,phone,social_links,paypal_me_link,paypal_enabled'])
            ->firstOrFail();

        if (!$plan->isAvailable()) {
            abort(404, 'Este plan no está disponible actualmente.');
        }

        // Get other active plans from this coach
        $otherPlans = CoachPlan::where('coach_id', $plan->coach_id)
            ->where('is_active', true)
            ->where('is_public', true)
            ->where('id', '!=', $plan->id)
            ->select(['id', 'name', 'slug', 'price', 'duration_days'])
            ->limit(4)
            ->get();

        // Extract social links
        $socialLinks = $plan->coach->social_links ?? [];

        // Transform results_images to expected format
        $results = collect($plan->results_images ?? [])->map(function ($result, $index) {
            return [
                'id' => $result['id'] ?? ($index + 1),
                'client_name' => $result['client_name'] ?? $result['name'] ?? 'Cliente',
                'before_image' => !empty($result['before']) ? asset('storage/' . $result['before']) : null,
                'after_image' => !empty($result['after']) ? asset('storage/' . $result['after']) : null,
                'transformation_duration' => $result['transformation_duration'] ?? $result['description'] ?? '',
                'testimonial' => $result['testimonial'] ?? null,
                'category' => $result['category'] ?? 'general',
            ];
        })->filter(function ($result) {
            // Only require before_image, after_image is optional
            return !empty($result['before_image']);
        })->values()->all();

        return Inertia::render('Public/CoachPlan', [
            'plan' => array_merge($plan->toArray(), [
                'featured_image' => $plan->hero_image ? asset('storage/' . $plan->hero_image) : ($plan->image ? asset('storage/' . $plan->image) : null),
            ]),
            'colors' => $plan->landing_colors ?? [],
            'coach' => [
                'name' => $plan->coach->name,
                'email' => $plan->coach->email,
                'profile_photo' => $plan->coach->avatar ? asset('storage/' . $plan->coach->avatar) : null,
                'bio' => $plan->coach->bio,
                'business_name' => $plan->coach->name, // Use name as fallback
                'credentials' => null,
                'whatsapp' => $plan->coach->phone,
                'instagram' => $socialLinks['instagram'] ?? null,
                'youtube' => $socialLinks['youtube'] ?? null,
                'clients_count' => $plan->coach->clients()->count(),
            ],
            'results' => $results,
            'paymentEnabled' => $plan->coach->paypal_enabled,
            'paymentLink' => $plan->getPaymentLink(),
            'otherPlans' => $otherPlans,
        ]);
    }

    /**
     * Display all results/transformations for a plan.
     */
    public function results(string $slug): Response
    {
        $plan = CoachPlan::where('slug', $slug)
            ->where('is_active', true)
            ->where('is_public', true)
            ->with(['coach:id,name,email,avatar,bio'])
            ->firstOrFail();

        return Inertia::render('Public/CoachPlanResults', [
            'plan' => [
                'id' => $plan->id,
                'name' => $plan->name,
                'slug' => $plan->slug,
                'price' => $plan->price,
            ],
            'coach' => [
                'name' => $plan->coach->name,
                'avatar' => $plan->coach->avatar,
            ],
            'results' => $plan->results_images ?? [],
        ]);
    }

    /**
     * Show purchase form (for authenticated clients).
     */
    public function purchase(Request $request, string $slug): Response
    {
        $plan = CoachPlan::where('slug', $slug)
            ->where('is_active', true)
            ->where('is_public', true)
            ->with(['coach:id,name,avatar,paypal_me_link,paypal_email,paypal_enabled'])
            ->firstOrFail();

        if (!$plan->isAvailable()) {
            abort(404, 'Este plan no está disponible actualmente.');
        }

        $user = $request->user();
        $client = null;

        // Check if user is a client of this coach
        if ($user && $user->role === 'client') {
            $client = $user->clientProfile;

            // Check for existing active purchase
            $existingPurchase = CoachPlanPurchase::where('coach_plan_id', $plan->id)
                ->where('client_id', $client?->id)
                ->whereIn('status', ['pending', 'paid'])
                ->where(function ($q) {
                    $q->whereNull('ends_at')->orWhere('ends_at', '>', now());
                })
                ->first();

            if ($existingPurchase) {
                return Inertia::render('Public/CoachPlanAlreadyPurchased', [
                    'plan' => $plan,
                    'purchase' => $existingPurchase,
                ]);
            }
        }

        return Inertia::render('Public/CoachPlanPurchase', [
            'plan' => $plan,
            'coach' => [
                'name' => $plan->coach->name,
                'avatar' => $plan->coach->avatar,
            ],
            'paymentLink' => $plan->getPaymentLink(),
            'client' => $client,
        ]);
    }

    /**
     * Process purchase request.
     */
    public function processPurchase(Request $request, string $slug): RedirectResponse
    {
        $plan = CoachPlan::where('slug', $slug)
            ->where('is_active', true)
            ->where('is_public', true)
            ->with('coach')
            ->firstOrFail();

        if (!$plan->isAvailable()) {
            return back()->with('error', 'Este plan no está disponible actualmente.');
        }

        $user = $request->user();

        if (!$user || $user->role !== 'client') {
            return back()->with('error', 'Debes ser un cliente registrado para comprar este plan.');
        }

        $client = $user->clientProfile;

        if (!$client) {
            return back()->with('error', 'No tienes un perfil de cliente asociado.');
        }

        // Check client is associated with this coach
        if ($client->user_id !== $plan->coach_id) {
            return back()->with('error', 'Solo los clientes de este entrenador pueden comprar este plan.');
        }

        $request->validate([
            'payment_method' => 'required|in:paypal,paypal_me,transfer,cash,other',
            'notes' => 'nullable|string|max:500',
        ]);

        // Check for existing pending purchase
        $existingPurchase = CoachPlanPurchase::where('coach_plan_id', $plan->id)
            ->where('client_id', $client->id)
            ->where('status', 'pending')
            ->first();

        if ($existingPurchase) {
            return redirect()->route('public.coach-plan.status', [
                'slug' => $slug,
                'purchase' => $existingPurchase->id,
            ])->with('info', 'Ya tienes una solicitud de compra pendiente.');
        }

        // Create purchase record
        $purchase = CoachPlanPurchase::create([
            'coach_plan_id' => $plan->id,
            'client_id' => $client->id,
            'amount' => $plan->price,
            'payment_method' => $request->payment_method,
            'notes' => $request->notes,
            'status' => 'pending',
        ]);

        // Redirect to PayPal if that method was selected
        if (in_array($request->payment_method, ['paypal', 'paypal_me']) && $plan->getPaymentLink()) {
            return redirect()->away($plan->getPaymentLink());
        }

        return redirect()->route('public.coach-plan.status', [
            'slug' => $slug,
            'purchase' => $purchase->id,
        ])->with('success', 'Solicitud de compra enviada. El entrenador confirmará tu pago.');
    }

    /**
     * Show purchase status.
     */
    public function purchaseStatus(Request $request, string $slug, CoachPlanPurchase $purchase): Response
    {
        $plan = CoachPlan::where('slug', $slug)->firstOrFail();

        // Verify purchase belongs to this plan
        if ($purchase->coach_plan_id !== $plan->id) {
            abort(404);
        }

        // Verify user owns this purchase
        $user = $request->user();
        if ($user && $user->role === 'client' && $user->clientProfile) {
            if ($purchase->client_id !== $user->clientProfile->id) {
                abort(403);
            }
        }

        return Inertia::render('Public/CoachPlanStatus', [
            'plan' => $plan,
            'purchase' => $purchase,
            'coach' => [
                'name' => $plan->coach->name,
            ],
        ]);
    }
}
