<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Coupon;
use App\Models\Address;
use App\Models\Bank;
use App\Models\Setting;
use App\Models\User;
use App\Models\PaymentFeeSetting;
use App\Notifications\OrderConfirmed;
use App\Notifications\AdminNewStoreOrder;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Notification;
use Inertia\Inertia;

class StoreController extends Controller
{
    protected function getLayout()
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        if ($user->role === 'admin') {
            return 'admin';
        } elseif ($user->role === 'coach') {
            return 'coach';
        }
        return 'client';
    }

    protected function routePrefix()
    {
        return $this->getLayout() . '.store';
    }

    public function index(Request $request)
    {
        $query = Product::with('category')
            ->active()
            ->when($request->search, function ($q, $search) {
                $q->where('name', 'like', "%{$search}%");
            })
            ->when($request->category, function ($q, $category) {
                $q->where('category_id', $category);
            })
            ->when($request->sort === 'price_asc', fn($q) => $q->orderBy('price', 'asc'))
            ->when($request->sort === 'price_desc', fn($q) => $q->orderBy('price', 'desc'))
            ->when($request->sort === 'newest', fn($q) => $q->latest())
            ->when($request->sort === 'popular', fn($q) => $q->orderBy('sales_count', 'desc'))
            ->when(!$request->sort, fn($q) => $q->orderBy('is_featured', 'desc')->latest());

        return Inertia::render('Store/Index', [
            'products' => $query->paginate(12)->withQueryString(),
            'categories' => ProductCategory::active()->ordered()->get(),
            'featuredProducts' => Product::active()->featured()->take(6)->get(),
            'filters' => $request->only(['search', 'category', 'sort']),
            'layout' => $this->getLayout(),
            'routePrefix' => $this->routePrefix(),
        ]);
    }

    public function show(Product $product)
    {
        // Increment view count
        $product->increment('view_count');

        $relatedProducts = Product::active()
            ->where('id', '!=', $product->id)
            ->when($product->category_id, function ($q, $catId) {
                $q->where('category_id', $catId);
            })
            ->take(4)
            ->get();

        // Load approved reviews with user info
        $reviews = $product->approvedReviews()
            ->with('user:id,name,avatar')
            ->latest()
            ->get()
            ->map(function ($review) {
                return [
                    'id' => $review->id,
                    'rating' => $review->rating,
                    'title' => $review->title,
                    'comment' => $review->comment,
                    'user_name' => $review->user->name ?? 'Cliente',
                    'user_avatar' => $review->user?->avatar_url,
                    'created_at' => $review->created_at->diffForHumans(),
                ];
            });

        // Check if current user already reviewed
        /** @var bool $isAuthenticated */
        $isAuthenticated = auth()->check();
        /** @var int|null $userId */
        $userId = auth()->id();
        $userReview = $isAuthenticated
            ? $product->reviews()->where('user_id', $userId)->first()
            : null;

        // Shipping zones
        $shippingZones = \App\Models\ShippingZone::active()->ordered()->get();

        return Inertia::render('Store/Show', [
            'product' => $product->load('category'),
            'relatedProducts' => $relatedProducts,
            'reviews' => $reviews,
            'userReview' => $userReview,
            'shippingZones' => $shippingZones,
            'layout' => $this->getLayout(),
            'routePrefix' => $this->routePrefix(),
        ]);
    }

    public function cart()
    {
        $cart = session()->get('cart', []);
        $items = [];
        $subtotal = 0;

        foreach ($cart as $productId => $quantity) {
            $product = Product::find($productId);
            if ($product) {
                $items[] = [
                    'product' => $product,
                    'quantity' => $quantity,
                    'subtotal' => $product->price * $quantity,
                ];
                $subtotal += $product->price * $quantity;
            }
        }

        return Inertia::render('Store/Cart', [
            'items' => $items,
            'subtotal' => $subtotal,
            'layout' => $this->getLayout(),
            'routePrefix' => $this->routePrefix(),
        ]);
    }

    public function addToCart(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'integer|min:1|max:10',
        ]);

        $product = Product::findOrFail($validated['product_id']);
        $quantity = $validated['quantity'] ?? 1;

        if ($product->track_stock && $product->stock < $quantity && !$product->allow_backorder) {
            return back()->with('error', 'No hay suficiente stock disponible.');
        }

        $cart = session()->get('cart', []);
        $cart[$product->id] = ($cart[$product->id] ?? 0) + $quantity;
        session()->put('cart', $cart);

        return back()->with('success', 'Producto agregado al carrito.');
    }

    public function updateCart(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'integer|min:0|max:10',
        ]);

        $cart = session()->get('cart', []);

        if ($validated['quantity'] <= 0) {
            unset($cart[$validated['product_id']]);
        } else {
            $cart[$validated['product_id']] = $validated['quantity'];
        }

        session()->put('cart', $cart);

        return back()->with('success', 'Carrito actualizado.');
    }

    public function removeFromCart(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        $cart = session()->get('cart', []);
        unset($cart[$validated['product_id']]);
        session()->put('cart', $cart);

        return back()->with('success', 'Producto eliminado del carrito.');
    }

    /**
     * Get available payment methods from admin settings
     */
    protected function getAvailablePaymentMethods(): array
    {
        $methods = [];

        // Cash on delivery - always available
        $methods[] = [
            'id' => 'cash',
            'name' => 'Pago contra entrega',
            'description' => 'Paga cuando recibas tu pedido',
            'icon' => 'cash',
        ];

        // Check Stripe
        if (Setting::get('stripe_enabled') && Setting::get('stripe_secret')) {
            $methods[] = [
                'id' => 'stripe',
                'name' => 'Tarjeta de Crédito/Débito',
                'description' => 'Pago seguro con Stripe',
                'icon' => 'credit-card',
            ];
        }

        // Check PayPal
        if (Setting::get('paypal_enabled') && Setting::get('paypal_client_id')) {
            $methods[] = [
                'id' => 'paypal',
                'name' => 'PayPal',
                'description' => 'Paga con tu cuenta PayPal',
                'icon' => 'paypal',
            ];
        }

        // Bank Transfer
        $banks = Bank::active()->ordered()->get();
        if ($banks->count() > 0) {
            $methods[] = [
                'id' => 'transfer',
                'name' => 'Transferencia Bancaria',
                'description' => 'Depósito o transferencia a nuestra cuenta',
                'icon' => 'bank',
                'banks' => $banks,
            ];
        }

        return $methods;
    }

    public function checkout()
    {
        $cart = session()->get('cart', []);
        if (empty($cart)) {
            return redirect()->route($this->routePrefix() . '.index')
                ->with('error', 'Tu carrito está vacío.');
        }

        $items = [];
        $subtotal = 0;

        foreach ($cart as $productId => $quantity) {
            $product = Product::find($productId);
            if ($product) {
                $items[] = [
                    'product' => $product,
                    'quantity' => $quantity,
                    'subtotal' => $product->price * $quantity,
                ];
                $subtotal += $product->price * $quantity;
            }
        }

        // Check if a coupon is applied in session
        $appliedCoupon = null;
        $discount = 0;
        $couponId = session()->get('coupon');
        if ($couponId) {
            $coupon = Coupon::find($couponId);
            /** @var int|null $userId */
            $userId = auth()->id();
            if ($coupon && $coupon->isValid() && $coupon->canBeUsedByUser($userId)) {
                $discount = $coupon->calculateDiscount($subtotal);
                $appliedCoupon = [
                    'id' => $coupon->id,
                    'code' => $coupon->code,
                    'name' => $coupon->name,
                    'type' => $coupon->type,
                    'value' => $coupon->value,
                    'discount' => $discount,
                ];
            } else {
                // Clear invalid coupon from session
                session()->forget('coupon');
            }
        }

        /** @var \App\Models\User $user */
        $user = auth()->user();
        $addresses = $user->addresses ?? collect();
        $paymentMethods = $this->getAvailablePaymentMethods();

        // Get payment fee settings
        $paymentFees = PaymentFeeSetting::active()->get()->mapWithKeys(function ($fee) {
            return [$fee->gateway => [
                'percentage' => $fee->percentage,
                'fixed_amount' => $fee->fixed_amount,
                'description' => $fee->fee_description,
                'pass_to_customer' => $fee->pass_to_customer,
            ]];
        });

        // Shipping zones
        $shippingZones = \App\Models\ShippingZone::active()->ordered()->get();
        $freeShippingThreshold = (float) Setting::get('store_free_shipping_threshold', 0);
        $freeShippingEnabled = Setting::get('store_enable_free_shipping', 'false') === 'true';

        return Inertia::render('Store/Checkout', [
            'items' => $items,
            'subtotal' => $subtotal,
            'discount' => $discount,
            'total' => $subtotal - $discount,
            'appliedCoupon' => $appliedCoupon,
            'addresses' => $addresses,
            'paymentMethods' => $paymentMethods,
            'paymentFees' => $paymentFees,
            'shippingZones' => $shippingZones,
            'freeShippingThreshold' => $freeShippingThreshold,
            'freeShippingEnabled' => $freeShippingEnabled,
            'layout' => $this->getLayout(),
            'routePrefix' => $this->routePrefix(),
        ]);
    }

    public function applyCoupon(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string',
        ]);

        $coupon = Coupon::where('code', strtoupper($validated['code']))->first();

        if (!$coupon) {
            return back()->with('error', 'Cupón no encontrado.');
        }

        if (!$coupon->isValid()) {
            return back()->with('error', 'Este cupón no es válido o ha expirado.');
        }

        /** @var int|null $userId */
        $userId = auth()->id();
        if (!$coupon->canBeUsedByUser($userId)) {
            return back()->with('error', 'Ya has usado este cupón el máximo número de veces permitido.');
        }

        session()->put('coupon', $coupon->id);

        return back()->with('success', 'Cupón aplicado correctamente.');
    }

    public function removeCoupon()
    {
        session()->forget('coupon');
        return back()->with('success', 'Cupón eliminado.');
    }

    public function processCheckout(Request $request)
    {
        $cart = session()->get('cart', []);
        if (empty($cart)) {
            return back()->with('error', 'Tu carrito está vacío.');
        }

        $validated = $request->validate([
            'shipping_address' => 'required|array',
            'shipping_address.first_name' => 'required|string|max:255',
            'shipping_address.last_name' => 'required|string|max:255',
            'shipping_address.address_line1' => 'required|string|max:255',
            'shipping_address.address_line2' => 'nullable|string|max:255',
            'shipping_address.city' => 'required|string|max:255',
            'shipping_address.state' => 'nullable|string|max:255',
            'shipping_address.postal_code' => 'nullable|string|max:20',
            'shipping_address.phone' => 'nullable|string|max:20',
            'payment_method' => 'required|string',
            'notes' => 'nullable|string|max:500',
            'save_address' => 'boolean',
        ]);

        // Calculate totals
        $subtotal = 0;
        $items = [];

        foreach ($cart as $productId => $quantity) {
            $product = Product::find($productId);
            if ($product) {
                $items[] = OrderItem::createFromProduct($product, $quantity);
                $subtotal += $product->price * $quantity;
            }
        }

        // Apply coupon if present
        $discount = 0;
        $couponCode = null;
        $couponId = session()->get('coupon');
        if ($couponId) {
            $coupon = Coupon::find($couponId);
            /** @var int|null $userId */
            $userId = auth()->id();
            if ($coupon && $coupon->isValid() && $coupon->canBeUsedByUser($userId)) {
                $discount = $coupon->calculateDiscount($subtotal);
                $couponCode = $coupon->code;
            }
        }

        $tax = 0; // TODO: Calculate based on products' tax rates
        $shipping = 0; // TODO: Calculate shipping

        // Calculate payment fee if applicable
        $paymentFee = 0;
        $paymentFeeMethod = null;
        $paymentFeeRate = null;

        if (in_array($validated['payment_method'], ['paypal', 'stripe'])) {
            $feeSetting = PaymentFeeSetting::getForGateway($validated['payment_method']);
            if ($feeSetting && $feeSetting->pass_to_customer) {
                $paymentFee = $feeSetting->calculateFee($subtotal - $discount);
                $paymentFeeMethod = $feeSetting->gateway;
                $paymentFeeRate = $feeSetting->percentage;
            }
        }

        $total = $subtotal - $discount + $tax + $shipping + $paymentFee;

        // Store always uses Dominican Pesos (DOP)
        $currency = 'DOP';
        $currencySymbol = 'RD$';

        // Create order
        /** @var int|null $userId */
        $userId = auth()->id();
        $order = Order::create([
            'user_id' => $userId,
            'subtotal' => $subtotal,
            'tax' => $tax,
            'shipping' => $shipping,
            'discount' => $discount,
            'payment_fee' => $paymentFee,
            'payment_fee_method' => $paymentFeeMethod,
            'payment_fee_rate' => $paymentFeeRate,
            'total' => $total,
            'currency' => $currency,
            'currency_symbol' => $currencySymbol,
            'coupon_code' => $couponCode,
            'payment_method' => $validated['payment_method'],
            'shipping_address' => $validated['shipping_address'],
            'notes' => $validated['notes'] ?? null,
            'status' => $validated['payment_method'] === 'cash' ? 'pending' : 'awaiting_payment',
        ]);

        // Save order items
        foreach ($items as $item) {
            $order->items()->save($item);
        }

        // Update product stock and sales
        foreach ($cart as $productId => $quantity) {
            $product = Product::find($productId);
            if ($product) {
                $product->decrementStock($quantity);
                $product->incrementSalesCount($quantity);
            }
        }

        // Track coupon usage
        if ($couponId) {
            $coupon = Coupon::find($couponId);
            if ($coupon) {
                $coupon->incrementUsage();
                /** @var int|null $userId */
                $userId = auth()->id();
                $order->couponUsage()->create([
                    'coupon_id' => $couponId,
                    'user_id' => $userId,
                ]);
            }
        }

        // Save address if requested
        if ($validated['save_address'] ?? false) {
            /** @var \App\Models\User $user */
            $user = auth()->user();
            $user->addresses()->create($validated['shipping_address']);
        }

        // Clear cart and coupon
        session()->forget(['cart', 'coupon']);

        // Send notification to customer
        $order->user->notify(new OrderConfirmed($order));

        // Send notification to all admins
        $admins = User::where('role', 'admin')->get();
        Notification::send($admins, new AdminNewStoreOrder($order));

        // Redirect based on payment method
        $paymentMethod = $validated['payment_method'];

        if ($paymentMethod === 'paypal') {
            // Redirect to PayPal payment
            return redirect()->route('store.payment.paypal', ['order' => $order->id]);
        }

        if ($paymentMethod === 'stripe') {
            // Redirect to Stripe payment
            return redirect()->route('store.payment.stripe', ['order' => $order->id]);
        }

        if ($paymentMethod === 'transfer') {
            // Show bank transfer instructions
            return redirect()->route($this->routePrefix() . '.orders.show', $order->id)
                ->with('success', '¡Pedido realizado! Por favor realiza la transferencia bancaria para completar tu pedido.');
        }

        // Cash on delivery - just show confirmation
        return redirect()->route($this->routePrefix() . '.orders.show', $order->id)
            ->with('success', '¡Pedido realizado con éxito! Tu número de orden es: ' . $order->order_number);
    }

    public function orders()
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        $orders = $user->orders()
            ->with('items')
            ->latest()
            ->paginate(10);

        return Inertia::render('Store/Orders/Index', [
            'orders' => $orders,
            'layout' => $this->getLayout(),
            'routePrefix' => $this->routePrefix(),
        ]);
    }

    public function showOrder(Order $order)
    {
        // Ensure user can only view their own orders
        /** @var int|null $userId */
        $userId = auth()->id();
        if ($order->user_id !== $userId) {
            abort(403);
        }

        // Get available payment methods for potential change
        $paymentMethods = $this->getAvailablePaymentMethods();

        return Inertia::render('Store/Orders/Show', [
            'order' => $order->load('items'),
            'layout' => $this->getLayout(),
            'routePrefix' => $this->routePrefix(),
            'paymentMethods' => $paymentMethods,
        ]);
    }

    public function changePaymentMethod(Request $request, Order $order)
    {
        // Ensure user can only modify their own orders
        /** @var int|null $userId */
        $userId = auth()->id();
        if ($order->user_id !== $userId) {
            abort(403);
        }

        // Only allow changes for pending orders that haven't been paid
        if ($order->status !== 'pending' || $order->payment_status === 'paid') {
            return back()->with('error', 'No se puede cambiar el método de pago de este pedido.');
        }

        $validated = $request->validate([
            'payment_method' => 'required|string',
        ]);

        $order->payment_method = $validated['payment_method'];
        $order->save();

        return back()->with('success', 'Método de pago actualizado correctamente.');
    }

    /**
     * Submit a product review.
     */
    public function submitReview(Request $request, Product $product)
    {
        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'title' => 'nullable|string|max:255',
            'comment' => 'nullable|string|max:1000',
        ]);

        // Check if user already reviewed this product
        /** @var int|null $userId */
        $userId = auth()->id();
        $existing = $product->reviews()->where('user_id', $userId)->first();
        if ($existing) {
            $existing->update($validated);
            return back()->with('success', 'Tu reseña ha sido actualizada.');
        }

        $product->reviews()->create([
            'user_id' => $userId,
            'rating' => $validated['rating'],
            'title' => $validated['title'] ?? null,
            'comment' => $validated['comment'] ?? null,
            'is_approved' => true, // Auto-approve for now
        ]);

        return back()->with('success', '¡Gracias por tu reseña!');
    }
}
