<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LandingTestimonial extends Model
{
    protected $fillable = [
        'coach_id',
        'name',
        'role',
        'content',
        'avatar',
        'rating',
        'sort_order',
        'is_active',
        'status',
        'source',
        'rejection_reason',
        'submitted_at',
        'reviewed_at',
        'reviewed_by',
    ];

    protected $casts = [
        'rating' => 'integer',
        'sort_order' => 'integer',
        'is_active' => 'boolean',
        'submitted_at' => 'datetime',
        'reviewed_at' => 'datetime',
    ];

    /**
     * Status constants.
     */
    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    /**
     * Source constants.
     */
    const SOURCE_ADMIN = 'admin';
    const SOURCE_COACH = 'coach';

    /**
     * Get the coach who submitted this testimonial.
     */
    public function coach(): BelongsTo
    {
        return $this->belongsTo(User::class, 'coach_id');
    }

    /**
     * Get the admin who reviewed this testimonial.
     */
    public function reviewer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    /**
     * Get active and approved testimonials ordered.
     */
    public static function getActive()
    {
        return self::where('is_active', true)
            ->where('status', self::STATUS_APPROVED)
            ->orderBy('sort_order')
            ->get();
    }

    /**
     * Get pending testimonials for admin review.
     */
    public static function getPending()
    {
        return self::where('status', self::STATUS_PENDING)
            ->orderBy('submitted_at', 'desc')
            ->get();
    }

    /**
     * Approve this testimonial.
     */
    public function approve(int $reviewerId): void
    {
        $this->update([
            'status' => self::STATUS_APPROVED,
            'is_active' => true,
            'reviewed_at' => now(),
            'reviewed_by' => $reviewerId,
            'sort_order' => self::max('sort_order') + 1,
        ]);
    }

    /**
     * Reject this testimonial.
     */
    public function reject(int $reviewerId, ?string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_REJECTED,
            'is_active' => false,
            'reviewed_at' => now(),
            'reviewed_by' => $reviewerId,
            'rejection_reason' => $reason,
        ]);
    }

    /**
     * Check if testimonial was submitted by a coach.
     */
    public function isFromCoach(): bool
    {
        return $this->source === self::SOURCE_COACH;
    }

    /**
     * Check if testimonial is pending review.
     */
    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Scope for pending testimonials.
     */
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    /**
     * Scope for approved testimonials.
     */
    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }

    /**
     * Scope for coach submissions.
     */
    public function scopeFromCoaches($query)
    {
        return $query->where('source', self::SOURCE_COACH);
    }
}
