<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Order extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'order_number',
        'user_id',
        'status',
        'subtotal',
        'tax',
        'shipping',
        'discount',
        'payment_fee',
        'payment_fee_method',
        'payment_fee_rate',
        'total',
        'currency',
        'currency_symbol',
        'coupon_code',
        'payment_method',
        'payment_status',
        'transaction_id',
        'shipping_address',
        'billing_address',
        'tracking_number',
        'tracking_url',
        'notes',
        'admin_notes',
        'paid_at',
        'shipped_at',
        'delivered_at',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'tax' => 'decimal:2',
        'shipping' => 'decimal:2',
        'discount' => 'decimal:2',
        'payment_fee' => 'decimal:2',
        'payment_fee_rate' => 'decimal:2',
        'total' => 'decimal:2',
        'shipping_address' => 'array',
        'billing_address' => 'array',
        'paid_at' => 'datetime',
        'shipped_at' => 'datetime',
        'delivered_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {
            if (empty($order->order_number)) {
                $order->order_number = 'ORD-' . strtoupper(Str::random(8));
            }
        });

        // Note: Status change notifications are handled by OrderController::sendStatusNotification()
        // to avoid duplicate notifications and to provide richer context (tracking info, cancellation reasons, etc.)
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function couponUsage()
    {
        return $this->hasOne(CouponUsage::class);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeProcessing($query)
    {
        return $query->where('status', 'processing');
    }

    public function scopeShipped($query)
    {
        return $query->where('status', 'shipped');
    }

    public function scopeDelivered($query)
    {
        return $query->where('status', 'delivered');
    }

    public function scopeCompleted($query)
    {
        return $query->whereIn('status', ['shipped', 'delivered']);
    }

    public function getStatusLabelAttribute()
    {
        $labels = [
            'pending' => 'Pendiente',
            'processing' => 'Procesando',
            'shipped' => 'Enviado',
            'delivered' => 'Entregado',
            'cancelled' => 'Cancelado',
            'refunded' => 'Reembolsado',
        ];

        return $labels[$this->status] ?? $this->status;
    }

    public function getStatusColorAttribute()
    {
        $colors = [
            'pending' => 'yellow',
            'processing' => 'blue',
            'shipped' => 'indigo',
            'delivered' => 'green',
            'cancelled' => 'red',
            'refunded' => 'gray',
        ];

        return $colors[$this->status] ?? 'gray';
    }

    public function getItemsCountAttribute()
    {
        return $this->items->sum('quantity');
    }

    public function updateStatus($status)
    {
        $this->status = $status;

        switch ($status) {
            case 'shipped':
                $this->shipped_at = now();
                break;
            case 'delivered':
                $this->delivered_at = now();
                break;
        }

        $this->save();
    }

    public function markAsPaid($transactionId = null)
    {
        $this->payment_status = 'paid';
        $this->paid_at = now();
        if ($transactionId) {
            $this->transaction_id = $transactionId;
        }
        $this->save();
    }

    /**
     * Format an amount using the order's currency.
     *
     * @param float $amount
     * @return string
     */
    public function formatCurrency(float $amount): string
    {
        $symbol = $this->currency_symbol ?? '$';
        $formatted = number_format($amount, 2);

        // Currencies that typically show symbol after the number
        $symbolAfter = ['EUR'];

        if (in_array($this->currency ?? 'USD', $symbolAfter)) {
            return "{$formatted}{$symbol}";
        }

        return "{$symbol}{$formatted}";
    }

    /**
     * Get the formatted total.
     *
     * @return string
     */
    public function getFormattedTotalAttribute(): string
    {
        return $this->formatCurrency($this->total);
    }

    /**
     * Get the formatted subtotal.
     *
     * @return string
     */
    public function getFormattedSubtotalAttribute(): string
    {
        return $this->formatCurrency($this->subtotal);
    }
}
