<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class ReferralSetting extends Model
{
    protected $fillable = [
        'key',
        'value',
        'type',
    ];

    /**
     * Get a setting value by key.
     */
    public static function get(string $key, $default = null)
    {
        $setting = self::where('key', $key)->first();
        
        if (!$setting) {
            return $default;
        }

        return self::castValue($setting->value, $setting->type);
    }

    /**
     * Set a setting value.
     */
    public static function set(string $key, $value, string $type = 'string'): void
    {
        self::updateOrCreate(
            ['key' => $key],
            ['value' => (string) $value, 'type' => $type]
        );

        Cache::forget('referral_settings');
    }

    /**
     * Get all settings as an array.
     */
    public static function getSettings(): array
    {
        return Cache::remember('referral_settings', 3600, function () {
            $settings = self::all()->pluck('value', 'key')->toArray();
            
            return [
                'enabled' => filter_var($settings['referral_enabled'] ?? true, FILTER_VALIDATE_BOOLEAN),
                'bonus_amount' => (float) ($settings['referral_bonus_amount'] ?? 10.00),
                'bonus_type' => $settings['referral_bonus_type'] ?? 'fixed',
                'qualification_type' => $settings['referral_qualification_type'] ?? 'first_payment',
                'min_subscription_amount' => (float) ($settings['referral_min_subscription_amount'] ?? 0),
            ];
        });
    }

    /**
     * Update multiple settings at once.
     */
    public static function updateSettings(array $data): void
    {
        foreach ($data as $key => $value) {
            $type = match($key) {
                'referral_enabled' => 'boolean',
                'referral_bonus_amount', 'referral_min_subscription_amount' => 'number',
                default => 'string',
            };
            
            self::set($key, $value, $type);
        }

        Cache::forget('referral_settings');
    }

    /**
     * Cast value based on type.
     */
    protected static function castValue($value, string $type)
    {
        return match($type) {
            'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'number' => (float) $value,
            'json' => json_decode($value, true),
            default => $value,
        };
    }
}
