<?php

namespace App\Notifications;

use App\Models\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent to administrators when an order is shipped.
 */
class AdminOrderShipped extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Order $order
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('📦 Pedido Enviado - #' . $this->order->order_number)
            ->greeting('Pedido enviado')
            ->line('Se ha marcado un pedido como enviado.')
            ->line('')
            ->line('**Detalles del Envío:**')
            ->line('**Orden:** #' . $this->order->order_number)
            ->line('**Cliente:** ' . $this->order->user->name)
            ->line('**Email:** ' . $this->order->user->email)
            ->line('**Fecha de Envío:** ' . $this->order->shipped_at?->format('d/m/Y H:i'));

        if ($this->order->tracking_number) {
            $message->line('**Número de Rastreo:** ' . $this->order->tracking_number);
        }

        if ($this->order->tracking_url) {
            $message->line('**URL de Rastreo:** ' . $this->order->tracking_url);
        }

        return $message
            ->line('')
            ->action('Ver Pedido', route('admin.store.orders.show', $this->order->id))
            ->salutation('Sistema de Notificaciones CoachPro');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'admin_order_shipped',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'customer_name' => $this->order->user->name,
            'tracking_number' => $this->order->tracking_number,
            'shipped_at' => $this->order->shipped_at?->toISOString(),
            'message' => "Pedido #{$this->order->order_number} enviado a {$this->order->user->name}",
            'title' => 'Pedido Enviado',
            'url' => route('admin.store.orders.show', $this->order->id),
            'icon' => 'truck',
            'priority' => 'normal',
        ];
    }

    public function databaseType(object $notifiable): string
    {
        return 'admin_order_shipped';
    }
}
