<?php

namespace App\Notifications;

use App\Models\Invoice;
use App\Models\Setting;
use App\Notifications\Traits\ChecksNotificationSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminPaymentReceived extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Invoice $invoice,
        public string $type = 'subscription' // 'subscription' or 'plan'
    ) {}

    /**
     * Get the notification's delivery channels.
     * Database first to ensure in-app notification is saved even if email fails.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $currency = Setting::get('currency', 'USD');
        $user = $this->invoice->user;

        $typeLabel = $this->type === 'subscription' ? 'Suscripción' : 'Plan de Coach';

        return (new MailMessage)
            ->subject("💰 Nuevo Pago Recibido - {$this->invoice->invoice_number}")
            ->greeting("¡Nuevo pago recibido!")
            ->line("Se ha completado un pago de **{$typeLabel}** en {$siteName}.")
            ->line('')
            ->line('**Detalles del Pago:**')
            ->line("Usuario: {$user->name} ({$user->email})")
            ->line("Factura: {$this->invoice->invoice_number}")
            ->line("Subtotal: \${$this->invoice->subtotal} {$currency}")
            ->line("Impuestos: \${$this->invoice->tax} {$currency}")
            ->line("**Total: \${$this->invoice->total} {$currency}**")
            ->line("Método: " . $this->getPaymentMethodName())
            ->line("Fecha: " . $this->invoice->paid_at?->format('d/m/Y H:i'))
            ->action('Ver Panel de Admin', route('admin.dashboard'))
            ->salutation("Sistema de Notificaciones - {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $typeLabel = $this->type === 'subscription' ? 'suscripción' : 'plan de coach';

        return [
            'type' => 'payment_received',
            'payment_type' => $this->type,
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'user_id' => $this->invoice->user_id,
            'user_name' => $this->invoice->user->name,
            'total' => $this->invoice->total,
            'payment_method' => $this->invoice->payment_method,
            'message' => "💰 Pago recibido: {$this->invoice->user->name} pagó \${$this->invoice->total} por {$typeLabel}",
            'title' => 'Nuevo pago recibido',
            'url' => route('admin.users.show', $this->invoice->user_id),
        ];
    }

    /**
     * Get human-readable payment method name.
     */
    protected function getPaymentMethodName(): string
    {
        return match($this->invoice->payment_method) {
            'stripe' => 'Tarjeta (Stripe)',
            'paypal' => 'PayPal',
            'bank_transfer' => 'Transferencia Bancaria',
            default => ucfirst($this->invoice->payment_method ?? 'Desconocido'),
        };
    }
}
