<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class CoachEmailChanged extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public string $oldEmail,
        public string $newEmail
    ) {}

    public function via($notifiable): array
    {
        // Solo enviar por email, no por base de datos
        return ['mail'];
    }

    public function toMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Bienvenido a tu nuevo email - CoachPro')
            ->greeting('¡Hola ' . $notifiable->name . '!')
            ->line('Te damos la bienvenida a tu nueva dirección de correo electrónico en CoachPro.')
            ->line('Tu email ha sido actualizado exitosamente de **' . $this->oldEmail . '** a **' . $this->newEmail . '**.')
            ->line('**Información importante:**')
            ->line('• Tu email anterior ya no será válido para iniciar sesión.')
            ->line('• Todos los correos de notificaciones se enviarán a esta nueva dirección.')
            ->line('• Si no realizaste este cambio, por favor contacta con soporte inmediatamente.')
            ->action('Acceder a mi perfil', url('/coach/profile'))
            ->line('Gracias por usar CoachPro para hacer crecer tu negocio de coaching.');
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'email_changed',
            'old_email' => $this->oldEmail,
            'new_email' => $this->newEmail,
            'message' => 'Tu email ha sido actualizado de ' . $this->oldEmail . ' a ' . $this->newEmail,
        ];
    }
}
