<?php

namespace App\Notifications;

use App\Models\Order;
use App\Notifications\Traits\ChecksNotificationSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent when order is marked as received by customer.
 */
class OrderReceived extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings;

    public function __construct(
        public Order $order
    ) {
    }

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('📦 Pedido Recibido #' . $this->order->order_number)
            ->greeting('¡Pedido Recibido!')
            ->line('Has confirmado la recepción de tu pedido.')
            ->line('')
            ->line('**Número de Orden:** ' . $this->order->order_number)
            ->line('**Total:** ' . $this->order->formatted_total)
            ->line('')
            ->line('Esperamos que disfrutes tu compra. Si tienes algún problema o pregunta, no dudes en contactarnos.')
            ->action('Ver Detalles del Pedido', route('store.orders.show', $this->order->id))
            ->salutation('Gracias por tu compra!');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'order_received',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'total' => $this->order->total,
            'message' => "Has confirmado la recepción del pedido #{$this->order->order_number}",
            'title' => 'Pedido Recibido',
            'url' => route('store.orders.show', $this->order->id),
            'icon' => 'check-circle',
        ];
    }

    public function databaseType(object $notifiable): string
    {
        return 'order_received';
    }
}
