<?php

namespace App\Notifications;

use App\Models\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent to customer when their order is refunded.
 */
class OrderRefunded extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Order $order,
        public float $refundAmount,
        public ?string $reason = null
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('Reembolso Procesado - Pedido #' . $this->order->order_number)
            ->greeting('Hola ' . $notifiable->name)
            ->line('Te informamos que se ha procesado un reembolso para tu pedido.')
            ->line('')
            ->line('**Detalles del Reembolso:**')
            ->line('• Número de Pedido: ' . $this->order->order_number)
            ->line('• Monto Reembolsado: ' . $this->order->formatCurrency($this->refundAmount))
            ->line('• Fecha de Procesamiento: ' . now()->format('d/m/Y H:i'));

        if ($this->reason) {
            $message->line('')
                ->line('**Motivo:**')
                ->line($this->reason);
        }

        $message->line('')
            ->line('**Tiempo de Procesamiento:**')
            ->line('El reembolso puede tardar entre 5-10 días hábiles en reflejarse en tu cuenta, dependiendo de tu institución financiera.')
            ->line('')
            ->action('Ver Detalles del Pedido', $this->getOrderUrl($notifiable))
            ->line('Si tienes preguntas sobre este reembolso, no dudes en contactarnos.')
            ->salutation('Gracias por tu comprensión.');

        return $message;
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'order_refunded',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'refund_amount' => $this->refundAmount,
            'reason' => $this->reason,
            'message' => "Reembolso de " . $this->order->formatCurrency($this->refundAmount) . " procesado para el pedido #{$this->order->order_number}",
            'title' => 'Reembolso Procesado',
            'url' => $this->getOrderUrl($notifiable),
        ];
    }

    protected function getOrderUrl(object $notifiable): string
    {
        $prefix = match($notifiable->role) {
            'admin' => 'admin',
            'coach' => 'coach',
            default => 'client',
        };

        return route("{$prefix}.store.orders.show", $this->order->id);
    }

    public function databaseType(object $notifiable): string
    {
        return 'order_refunded';
    }
}
