<?php

namespace App\Notifications;

use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionExpired extends Notification implements ShouldQueue
{
    use Queueable;

    protected Subscription $subscription;
    protected int $gracePeriodDays;

    public function __construct(Subscription $subscription, int $gracePeriodDays = 0)
    {
        $this->subscription = $subscription;
        $this->gracePeriodDays = $gracePeriodDays;
    }

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('Tu suscripción ha expirado')
            ->greeting('Hola ' . $notifiable->name)
            ->line('Tu suscripción al plan **' . $this->subscription->plan->name . '** ha expirado.');

        if ($this->gracePeriodDays > 0) {
            $message->line('Tienes un período de gracia de **' . $this->gracePeriodDays . ' días** para renovar sin perder acceso a tus datos.');
        } else {
            $message->line('Tu acceso a las funcionalidades premium ha sido desactivado.');
        }

        return $message
            ->line('Renueva ahora para continuar gestionando a tus clientes.')
            ->action('Renovar Suscripción', url('/coach/subscription'))
            ->line('¿Necesitas ayuda? Contáctanos en soporte@coachpro.com');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'subscription_expired',
            'subscription_id' => $this->subscription->id,
            'plan_name' => $this->subscription->plan->name,
            'grace_period_days' => $this->gracePeriodDays,
            'message' => '⛔ Tu suscripción al plan ' . $this->subscription->plan->name . ' ha expirado.',
            'title' => 'Suscripción expirada',
            'url' => route('coach.subscription.index'),
        ];
    }
}
