<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionExtended extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Subscription $subscription,
        public int $days
    ) {}

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $planName = $this->subscription->plan->name ?? 'Tu plan';
        $newEndDate = $this->subscription->end_date?->format('d/m/Y') ?? 'N/A';

        return (new MailMessage)
            ->subject("Suscripción Extendida - {$this->days} días adicionales")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("¡Buenas noticias! Tu suscripción al plan **{$planName}** ha sido extendida.")
            ->line("**Días añadidos:** {$this->days}")
            ->line("**Nueva fecha de vencimiento:** {$newEndDate}")
            ->action('Ver Mi Suscripción', route('coach.subscription.index'))
            ->salutation("Saludos,\nEl equipo de {$siteName}");
    }

    public function toArray(object $notifiable): array
    {
        $planName = $this->subscription->plan->name ?? 'Tu plan';
        $newEndDate = $this->subscription->end_date?->format('d/m/Y') ?? 'N/A';

        return [
            'type' => 'subscription_extended',
            'subscription_id' => $this->subscription->id,
            'days_added' => $this->days,
            'new_end_date' => $newEndDate,
            'message' => "🎉 Suscripción extendida {$this->days} días. Vence: {$newEndDate}",
            'title' => 'Suscripción extendida',
            'url' => route('coach.subscription.index'),
        ];
    }
}
