<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\WalletTransaction;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WalletTransactionRejected extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public WalletTransaction $transaction,
        public string $reason = 'Rechazado por administrador'
    ) {}

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $currency = Setting::get('currency', 'DOP');
        $amount = number_format(abs($this->transaction->amount), 2);
        $type = $this->transaction->type === 'withdrawal' ? 'retiro' : 'depósito';

        return (new MailMessage)
            ->subject("Transacción Rechazada - {$type} de \${$amount}")
            ->greeting("Hola {$notifiable->name},")
            ->line("Tu solicitud de {$type} ha sido rechazada.")
            ->line("**Monto:** \${$amount} {$currency}")
            ->line("**Razón:** {$this->reason}")
            ->line($this->transaction->type === 'withdrawal'
                ? "El monto ha sido devuelto a tu saldo."
                : "Por favor contacta soporte si tienes preguntas.")
            ->action('Ver Mi Billetera', route('coach.wallet.index'))
            ->salutation("Saludos,\nEl equipo de {$siteName}");
    }

    public function toArray(object $notifiable): array
    {
        $amount = number_format(abs($this->transaction->amount), 2);
        $type = $this->transaction->type === 'withdrawal' ? 'retiro' : 'depósito';

        return [
            'type' => 'wallet_transaction_rejected',
            'transaction_id' => $this->transaction->id,
            'transaction_type' => $this->transaction->type,
            'amount' => abs($this->transaction->amount),
            'reason' => $this->reason,
            'message' => "❌ {$type} de \${$amount} rechazado: {$this->reason}",
            'title' => 'Transacción rechazada',
            'url' => route('coach.wallet.index'),
        ];
    }
}
