<?php

namespace App\Services;

use App\Models\EmailTemplate;
use App\Models\CoachEmailTemplate;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

/**
 * Shared service for email template operations.
 * Consolidates common logic between Admin and Coach email template controllers.
 */
class EmailTemplateService
{
    /**
     * Get sample data for template previews.
     */
    public function getSampleData(): array
    {
        return [
            // General
            'app_name' => config('app.name', 'CoachPro'),
            'app_url' => config('app.url', 'http://localhost'),
            
            // User info
            'user_name' => 'Usuario Demo',
            'client_name' => 'Juan Pérez',
            'coach_name' => 'Carlos García',
            'buyer_name' => 'María López',
            'email' => 'demo@ejemplo.com',
            'password' => 'temporal123',
            
            // URLs
            'login_url' => url('/login'),
            'dashboard_url' => url('/coach/dashboard'),
            'renew_url' => url('/coach/subscription'),
            'invoice_url' => url('/coach/subscription/invoices'),
            
            // Plan info
            'plan_name' => 'Plan Premium',
            'amount' => '99.00',
            'currency' => 'USD',
            'start_date' => now()->format('d/m/Y'),
            'end_date' => now()->addDays(30)->format('d/m/Y'),
            'days_remaining' => '7',
            
            // Invoice info
            'invoice_number' => 'INV-202602-0001',
            'invoice_total' => '$99.00',
            'invoice_status' => 'Pendiente',
            'payment_date' => now()->format('d/m/Y'),
            
            // Payment
            'payment_instructions' => 'Transfiera a la cuenta bancaria indicada.',
            
            // Referral
            'referral_bonus' => '$50.00',
            'referred_name' => 'Ana García',
            
            // Content
            'message_preview' => 'Hola, tengo una pregunta sobre...',
            'workout_name' => 'Rutina de Fuerza A',
            'meal_plan_name' => 'Plan de Alimentación Semana 1',
        ];
    }

    /**
     * Replace template variables with data.
     */
    public function replaceVariables(string $content, array $data): string
    {
        foreach ($data as $key => $value) {
            $content = str_replace("{{{$key}}}", $value, $content);
        }
        return $content;
    }

    /**
     * Render a template with given data.
     */
    public function render(string $subject, string $body, array $data = []): array
    {
        $allData = array_merge($this->getSampleData(), $data);
        
        return [
            'subject' => $this->replaceVariables($subject, $allData),
            'body' => $this->replaceVariables($body, $allData),
        ];
    }

    /**
     * Send a test email.
     */
    public function sendTestEmail(string $to, string $subject, string $body, array $data = []): bool
    {
        $rendered = $this->render($subject, $body, $data);
        
        try {
            Mail::html($rendered['body'], function ($message) use ($to, $rendered) {
                $message->to($to)
                    ->subject('[TEST] ' . $rendered['subject']);
            });
            
            Log::info("Test email sent successfully to {$to}");
            return true;
        } catch (\Exception $e) {
            Log::error("Failed to send test email to {$to}: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get global email template by slug.
     */
    public function getGlobalTemplate(string $slug): ?EmailTemplate
    {
        return EmailTemplate::getBySlug($slug);
    }

    /**
     * Get coach-specific template or default.
     */
    public function getCoachTemplate(int $coachId, string $type): ?CoachEmailTemplate
    {
        return CoachEmailTemplate::getForCoach($coachId, $type);
    }

    /**
     * Validate email configuration.
     */
    public function validateMailConfig(): array
    {
        $issues = [];
        
        $mailer = config('mail.default');
        
        if ($mailer === 'log') {
            $issues[] = [
                'type' => 'warning',
                'message' => 'El mailer está configurado en modo "log". Los emails no se enviarán realmente.',
            ];
        }
        
        if ($mailer === 'smtp') {
            $host = config('mail.mailers.smtp.host');
            $port = config('mail.mailers.smtp.port');
            
            if (empty($host) || $host === '127.0.0.1') {
                $issues[] = [
                    'type' => 'error',
                    'message' => 'MAIL_HOST no está configurado correctamente.',
                ];
            }
            
            if (empty(config('mail.mailers.smtp.username'))) {
                $issues[] = [
                    'type' => 'error',
                    'message' => 'MAIL_USERNAME no está configurado.',
                ];
            }
        }
        
        $fromAddress = config('mail.from.address');
        if ($fromAddress === 'hello@example.com') {
            $issues[] = [
                'type' => 'warning',
                'message' => 'MAIL_FROM_ADDRESS usa el valor por defecto.',
            ];
        }
        
        return $issues;
    }
}
