<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Referral codes for coaches
        Schema::create('referral_codes', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade'); // coach who owns the code
            $table->string('code', 20)->unique();
            $table->integer('uses')->default(0);
            $table->integer('max_uses')->nullable(); // null = unlimited
            $table->boolean('is_active')->default(true);
            $table->timestamp('expires_at')->nullable();
            $table->timestamps();
            
            $table->index('code');
            $table->index(['user_id', 'is_active']);
        });

        // Referrals tracking
        Schema::create('referrals', function (Blueprint $table) {
            $table->id();
            $table->foreignId('referrer_id')->constrained('users')->onDelete('cascade'); // coach who referred
            $table->foreignId('referred_id')->constrained('users')->onDelete('cascade'); // new coach who signed up
            $table->foreignId('referral_code_id')->nullable()->constrained()->onDelete('set null');
            $table->enum('status', ['pending', 'qualified', 'rewarded', 'cancelled'])->default('pending');
            $table->decimal('reward_amount', 10, 2)->nullable();
            $table->foreignId('wallet_transaction_id')->nullable()->constrained()->onDelete('set null');
            $table->timestamp('qualified_at')->nullable();
            $table->timestamp('rewarded_at')->nullable();
            $table->timestamps();
            
            $table->unique('referred_id'); // a user can only be referred once
            $table->index(['referrer_id', 'status']);
        });

        // Referral settings (stored in settings table, but we create a dedicated one for complex config)
        Schema::create('referral_settings', function (Blueprint $table) {
            $table->id();
            $table->string('key')->unique();
            $table->text('value')->nullable();
            $table->string('type')->default('string'); // string, number, boolean, json
            $table->timestamps();
        });

        // Insert default referral settings
        DB::table('referral_settings')->insert([
            [
                'key' => 'referral_enabled',
                'value' => '1',
                'type' => 'boolean',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'referral_bonus_amount',
                'value' => '10.00',
                'type' => 'number',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'referral_bonus_type',
                'value' => 'fixed', // fixed or percentage
                'type' => 'string',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'referral_qualification_type',
                'value' => 'first_payment', // signup, first_payment, first_month
                'type' => 'string',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'referral_min_subscription_amount',
                'value' => '0',
                'type' => 'number',
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ]);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('referrals');
        Schema::dropIfExists('referral_codes');
        Schema::dropIfExists('referral_settings');
    }
};
