<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Create diet_plans table
        Schema::create('diet_plans', function (Blueprint $table) {
            $table->id();
            $table->foreignId('meal_plan_id')->constrained()->onDelete('cascade');
            $table->string('name');
            $table->text('description')->nullable();
            $table->json('days')->nullable(); // Array of weekdays: ['LUN', 'MAR', ...]
            $table->boolean('use_percentages')->default(true);
            $table->integer('protein_percent')->default(30);
            $table->integer('fat_percent')->default(25);
            $table->integer('carb_percent')->default(45);
            $table->integer('calorie_adjustment')->default(0); // -100 to +100
            $table->boolean('is_active')->default(true);
            $table->integer('order')->default(0);
            $table->timestamps();

            $table->index('meal_plan_id');
        });

        // Add diet_plan_id to meal_items
        Schema::table('meal_items', function (Blueprint $table) {
            $table->foreignId('diet_plan_id')->nullable()->after('meal_plan_id');
        });

        // Migrate existing data: Create one DietPlan per existing MealPlan
        $mealPlans = DB::table('meal_plans')->get();
        
        foreach ($mealPlans as $mealPlan) {
            // Create a DietPlan for this MealPlan
            $dietPlanId = DB::table('diet_plans')->insertGetId([
                'meal_plan_id' => $mealPlan->id,
                'name' => $mealPlan->name,
                'description' => $mealPlan->description,
                'days' => json_encode([]),
                'use_percentages' => true,
                'protein_percent' => 30,
                'fat_percent' => 25,
                'carb_percent' => 45,
                'calorie_adjustment' => 0,
                'is_active' => true,
                'order' => 0,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Update all meal_items for this meal_plan to point to the new diet_plan
            DB::table('meal_items')
                ->where('meal_plan_id', $mealPlan->id)
                ->update(['diet_plan_id' => $dietPlanId]);
        }

        // Add foreign key constraint after data migration
        Schema::table('meal_items', function (Blueprint $table) {
            $table->foreign('diet_plan_id')->references('id')->on('diet_plans')->onDelete('cascade');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('meal_items', function (Blueprint $table) {
            $table->dropForeign(['diet_plan_id']);
            $table->dropColumn('diet_plan_id');
        });

        Schema::dropIfExists('diet_plans');
    }
};
